# -*- coding: utf-8 -*-
"""
/***************************************************************************
 KictRainPredictorDialog
                                 A QGIS plugin
 A QGIS plugin for quantitative precipitation forecasting using deep learning models. This plugin utilizes radar data to predict rainfall patterns up to 180 minutes ahead with three different model options: standard Keras model, TFLite optimized model, and ensemble TFLite models. Developed based on the Korean Institute of Civil Engineering and Building Technology (KICT) rainfall prediction system.
 Generated by Plugin Builder: http://g-sherman.github.io/Qgis-Plugin-Builder/
                             -------------------
        begin                : 2025-07-10
        git sha              : $Format:%H$
        copyright            : (C) 2025 by KICT, HermeSys
        email                : sukmin28@hermesys.co.kr
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/
"""

import glob
import os
import threading
import time

import gdown
from PyQt5 import QtCore, QtWidgets
from qgis.core import QgsProject, QgsRasterLayer
from qgis.PyQt.QtCore import QObject, pyqtSignal
from qgis.PyQt.QtWidgets import (
    QApplication,
    QDialog,
    QFileDialog,
    QLabel,
    QMessageBox,
    QProgressBar,
    QProgressDialog,
    QPushButton,
    QVBoxLayout,
)

from kict_rain_forecast.kict_rain_forecast_dialog_base import Ui_Dialog


# 다운로드 관리자 클래스 - 스레드 간 통신을 위한 시그널 정의
class DownloadManager(QObject):
    # 시그널 정의
    create_progress_dialog_signal = pyqtSignal(str, int)
    update_progress_signal = pyqtSignal(int, str)
    update_dialog_title_signal = pyqtSignal(str)
    download_completed_signal = pyqtSignal(bool, str)
    download_error_signal = pyqtSignal(str)

    def __init__(self, parent=None):
        super(DownloadManager, self).__init__(parent)
        self.parent = parent
        self.canceled = False
        # 파일 다운로드 현황 관리를 위한 변수
        self.file_size = 0
        self.downloaded = 0
        self.start_time = 0
        self.progress_active = False


class KictRainPredictorDialog(QtWidgets.QDialog, Ui_Dialog):
    # 모델 다운로드 URL
    SINGLE_TARGET_MODEL_URL = (
        "https://drive.google.com/uc?id=1CxbdCAe8kRBqGQNEXVd-dHkKL8ISi_dq"
    )
    MULTI_TARGET_MODEL_URLS = {
        # 10분 간격으로 10분~180분까지의 모델 URL
        10: "https://drive.google.com/uc?id=14Cz1yDCtrbI3KoHlU4GiNXwyCVKS-clX",
        20: "https://drive.google.com/uc?id=11FNN6ekYG5gpmONQCozPQYc2FH-zLNlM",
        30: "https://drive.google.com/uc?id=1zsQjoh_nqqEa-9fz91P24xx1FoyDKB_G",
        40: "https://drive.google.com/uc?id=1FvcROg3Sal2NBAKXG3qRoPsVyIGfph5q",
        50: "https://drive.google.com/uc?id=1pixZOvE87vFUHLDdei-yvCisBuUJuthM",
        60: "https://drive.google.com/uc?id=13YWC2efMsKvbNpZ5L_daGkJ0Bpujezqc",
        70: "https://drive.google.com/uc?id=1MjzRQb1FWz0lKoaAGhiS23TvIjYncGB7",
        80: "https://drive.google.com/uc?id=1JUrlUDe1EYvoOkQj1jKRerLu5a4F3-as",
        90: "https://drive.google.com/uc?id=1xzxRDR_YZWz-2oiPEpjTUKQAUHe4-PHj",
        100: "https://drive.google.com/uc?id=1HlPbS1hlFruMS9uNAjUp20WmHtMoiSwR",
        110: "https://drive.google.com/uc?id=1DK2YSAKqDOWBhlH9BkfD0oJX2yzhnG7Z",
        120: "https://drive.google.com/uc?id=1t4KnIHIn3mWChzpJh4v20CB9lhsF84v6",
        130: "https://drive.google.com/uc?id=14HPeJjzE50ziTQOWs4HhwiK_ftHYIeq3",
        140: "https://drive.google.com/uc?id=1Ep721RZV3CpFy23se47jG8m9qQiygxAb",
        150: "https://drive.google.com/uc?id=1pCeW1umNiudMD7yWDHF-hfgb5sXW3YKC",
        160: "https://drive.google.com/uc?id=1GxzveX6pQeqqoVuvdj2qUeURJOgDTffp",
        170: "https://drive.google.com/uc?id=1h33EX7ZUlw-OJzgGUHKC6xYQkhJsoUWW",
        180: "https://drive.google.com/uc?id=1oHvK5CqXKVbbVW5eQtfyzoAp_RPaouZC",
    }

    def __init__(self, parent=None):
        """Constructor."""
        super(KictRainPredictorDialog, self).__init__(parent)
        # Set up the user interface from Designer through FORM_CLASS.
        # After self.setupUi() you can access any designer object by doing
        # self.<objectname>, and you can use autoconnect slots - see
        # http://qt-project.org/doc/qt-4.8/designer-using-a-ui-file.html

        # 최근 사용한 디렉토리 경로 저장 변수
        self.last_used_directory = os.path.expanduser("~")
        # #widgets-and-dialogs-with-auto-connect
        self.setupUi(self)

        # Cancel 버튼을 클릭하면 창을 닫도록 연결
        self.pushButton_2.clicked.connect(self.reject)

        # 모델 디렉토리 설정
        self.plugin_dir = os.path.dirname(os.path.abspath(__file__))
        self.models_dir = os.path.join(self.plugin_dir, "models")
        self.ensemble_dir = os.path.join(self.models_dir, "ensemble")

        # 다운로드 관리자 생성
        self.download_manager = DownloadManager(self)
        self.download_manager.create_progress_dialog_signal.connect(
            self.create_progress_dialog
        )
        self.download_manager.update_progress_signal.connect(
            self.update_progress_dialog
        )
        self.download_manager.update_dialog_title_signal.connect(
            self.update_progress_dialog_title
        )
        self.download_manager.download_completed_signal.connect(self.download_completed)
        self.download_manager.download_error_signal.connect(self.download_error)

        # 진행 상황 대화 상자 및 다운로드 취소 플래그
        self.progress_dialog = None

        # 버튼 연결
        self.pushButton_5.clicked.connect(self.select_output_folder)
        self.pushButton_download.clicked.connect(self.download_models)
        self.pushButton.clicked.connect(self.run_prediction)

        # 파일 선택 버튼 연결
        self.pushButton_file1.clicked.connect(lambda: self.select_input_file(1))
        self.pushButton_file2.clicked.connect(lambda: self.select_input_file(2))
        self.pushButton_file3.clicked.connect(lambda: self.select_input_file(3))
        self.pushButton_file4.clicked.connect(lambda: self.select_input_file(4))
        self.pushButton_batch.clicked.connect(self.select_batch_files)

        # 모델 설치 상태 확인
        self.check_model_installation()
        self.radioButton.toggled.connect(self.check_model_installation)
        self.radioButton_2.toggled.connect(self.check_model_installation)

        # 초기 모델 설치 상태 확인
        self.check_model_installation()

    def select_output_folder(self):
        """출력 폴더 선택 대화상자를 표시합니다."""
        folder = QFileDialog.getExistingDirectory(
            self, "출력 폴더 선택", self.last_used_directory
        )
        if folder:
            self.lineEdit_5.setText(folder)
            self.last_used_directory = folder

    def select_input_file(self, file_num):
        """입력 파일 선택 대화상자를 표시합니다.

        Args:
            file_num: 파일 번호 (1-4)
        """
        file_filter = "ASC 파일 (*.asc);;모든 파일 (*.*)"
        file_path, _ = QFileDialog.getOpenFileName(
            self, f"입력 파일 {file_num} 선택", self.last_used_directory, file_filter
        )
        if file_path:
            # 파일 경로에서 디렉토리 부분 저장
            self.last_used_directory = os.path.dirname(file_path)

            # 해당 comboBox에 파일 경로 추가
            if file_num == 1:
                combo_box = self.comboBox
            else:
                combo_box = getattr(self, f"comboBox_{file_num}")

            # 이미 목록에 있는지 확인
            file_name = os.path.basename(file_path)
            found = False
            for i in range(combo_box.count()):
                if combo_box.itemText(i) == file_name:
                    combo_box.setCurrentIndex(i)
                    found = True
                    break

            # 목록에 없으면 추가
            if not found:
                combo_box.addItem(file_name)
                combo_box.setCurrentText(file_name)

            # 전체 경로를 데이터로 저장
            combo_box.setItemData(combo_box.currentIndex(), file_path)

    def select_batch_files(self):
        """일괄 파일 선택 대화상자를 표시합니다."""
        folder = QFileDialog.getExistingDirectory(
            self, "레이더 데이터 폴더 선택", self.last_used_directory
        )
        if not folder:
            return

        self.last_used_directory = folder

        # 폴더 내의 모든 .asc 파일 찾기
        asc_files = sorted(glob.glob(os.path.join(folder, "*.asc")))

        if not asc_files:
            QMessageBox.warning(
                self, "파일 없음", f"{folder} 폴더에 ASC 파일이 없습니다."
            )
            return

        # 파일이 4개 이상인 경우 선택 대화상자 표시
        if len(asc_files) >= 4:
            # 시간순으로 정렬된 파일 목록에서 최근 4개 선택
            selected_files = asc_files[-4:]

            # 파일 순서 확인 메시지
            msg = "다음 파일들이 선택되었습니다:\n\n"
            msg += f"30분 전 데이터: {os.path.basename(selected_files[0])}\n"
            msg += f"20분 전 데이터: {os.path.basename(selected_files[1])}\n"
            msg += f"10분 전 데이터: {os.path.basename(selected_files[2])}\n"
            msg += f"예측 시점 데이터: {os.path.basename(selected_files[3])}\n\n"
            msg += "파일 순서가 올바른지 확인하세요."

            QMessageBox.information(self, "파일 선택 완료", msg)

            # 각 comboBox에 파일 설정
            combo_boxes = [
                self.comboBox,
                self.comboBox_2,
                self.comboBox_3,
                self.comboBox_4,
            ]
            for i, file_path in enumerate(selected_files):
                file_name = os.path.basename(file_path)

                # 이미 목록에 있는지 확인
                combo_box = combo_boxes[i]
                found = False
                for j in range(combo_box.count()):
                    if combo_box.itemText(j) == file_name:
                        combo_box.setCurrentIndex(j)
                        combo_box.setItemData(j, file_path)
                        found = True
                        break

                # 목록에 없으면 추가
                if not found:
                    combo_box.addItem(file_name)
                    combo_box.setCurrentText(file_name)
                    combo_box.setItemData(combo_box.currentIndex(), file_path)
        else:
            QMessageBox.warning(
                self,
                "파일 부족",
                f"{folder} 폴더에 ASC 파일이 4개 미만입니다.\n"
                "4개 이상의 파일이 필요합니다.",
            )
        if folder:
            self.lineEdit_5.setText(folder)

    def download_models(self):
        """선택된 모델 유형에 따라 모델을 다운로드합니다."""
        # 모델 디렉토리가 없으면 생성
        if not os.path.exists(self.models_dir):
            os.makedirs(self.models_dir, exist_ok=True)

        # 앙상블 모델 디렉토리가 없으면 생성
        if not os.path.exists(self.ensemble_dir):
            os.makedirs(self.ensemble_dir, exist_ok=True)

        # 다운로드 중 UI 비활성화
        self.setEnabled(False)
        self.pushButton_download.setText("다운로드 중...")

        # 선택된 모델 유형에 따라 다운로드 시작
        if self.radioButton.isChecked():  # Multi Target
            threading.Thread(target=self.download_multi_target_models).start()
        else:  # Single Target
            threading.Thread(target=self.download_single_target_model).start()

    def download_single_target_model(self):
        """단일 모델(Single Target)을 다운로드합니다."""
        # 다운로드 취소 플래그 초기화
        self.download_manager.canceled = False

        # 시그널을 통해 진행 상황 대화 상자 생성 요청
        self.download_manager.create_progress_dialog_signal.emit(
            "Single Target 모델 다운로드 중", 1
        )

        try:
            output_path = os.path.join(self.models_dir, "model-best.tflite")
            success = self.download_with_progress(
                self.SINGLE_TARGET_MODEL_URL, output_path, "Single Target 모델"
            )

            # 시그널을 통해 완료 알림
            self.download_manager.download_completed_signal.emit(
                success,
                "Single Target 모델 다운로드가 완료되었습니다."
                if success
                else "다운로드가 취소되었습니다.",
            )
        except Exception as e:
            # 오류 발생 시 시그널 발생
            self.download_manager.download_completed_signal.emit(
                False, f"다운로드 중 오류가 발생했습니다: {str(e)}"
            )

    def download_multi_target_models(self):
        """다중 모델(Multi Target)을 다운로드합니다."""
        # 총 모델 수
        total_models = len(self.MULTI_TARGET_MODEL_URLS)

        # 다운로드 취소 플래그 초기화
        self.download_manager.canceled = False

        # 시그널을 통해 진행 상황 대화 상자 생성 요청
        self.download_manager.create_progress_dialog_signal.emit(
            "Multi Target 모델 다운로드 중", total_models
        )

        try:
            # 18개 모델 다운로드
            success = True
            for i, (minutes, url) in enumerate(self.MULTI_TARGET_MODEL_URLS.items(), 1):
                # 취소 확인
                if self.download_manager.canceled:
                    success = False
                    break

                output_path = os.path.join(
                    self.ensemble_dir, f"model-best_fcst_{minutes}min.tflite"
                )
                model_name = f"Multi Target {minutes}min 모델"

                # 시그널을 통해 현재 다운로드 중인 모델 정보 업데이트
                self.download_manager.update_dialog_title_signal.emit(
                    f"Multi Target 모델 다운로드 중 ({i}/{total_models}): {minutes}min"
                )

                # 모델 다운로드
                if not self.download_with_progress(
                    url, output_path, model_name, total_models, i
                ):
                    success = False
                    break

            # 시그널을 통해 완료 알림
            self.download_manager.download_completed_signal.emit(
                success,
                "Multi Target 모델 다운로드가 완료되었습니다."
                if success
                else "다운로드가 취소되었습니다.",
            )
        except Exception as e:
            # 오류 발생 시 시그널 발생
            self.download_manager.download_completed_signal.emit(
                False, f"다운로드 중 오류가 발생했습니다: {str(e)}"
            )

    def download_with_progress(
        self, url, output_path, model_name, total_models=1, current_model=1
    ):
        """진행 상황 표시와 함께 모델을 다운로드합니다.

        Args:
            url: 다운로드할 URL
            output_path: 저장할 경로
            model_name: 모델 이름 (진행 상황 창에 표시)
            total_models: 전체 모델 수
            current_model: 현재 다운로드 중인 모델 번호
        """
        # 진행 상황 모니터링을 위한 변수
        self.download_manager.file_size = 0
        self.download_manager.downloaded = 0
        self.download_manager.start_time = time.time()
        self.download_manager.progress_active = True
        
        # 출력 디렉토리 확인 및 생성
        output_dir = os.path.dirname(output_path)
        if not os.path.exists(output_dir):
            os.makedirs(output_dir, exist_ok=True)

        # 파일 크기를 추정하기 위한 함수
        def estimate_file_size():
            try:
                # URL에서 리다이렉트를 확인하여 실제 파일 URL 찾기
                from urllib.parse import parse_qs, urlparse

                import requests

                # Google Drive 파일 ID 추출
                parsed = urlparse(url)
                if "drive.google.com" in parsed.netloc:
                    file_id = parse_qs(parsed.query).get("id", [""])[0]
                    if not file_id and "id=" in url:
                        # URL에서 id 직접 추출 시도
                        file_id = url.split("id=")[1].split("&")[0]

                    if file_id:
                        # 리다이렉트 URL 찾기
                        session = requests.Session()
                        response = session.get(
                            f"https://drive.google.com/uc?id={file_id}&export=download",
                            stream=True,
                        )
                        if response.status_code == 200:
                            try:
                                content_length = int(
                                    response.headers.get("content-length", 0)
                                )
                                if content_length > 0:
                                    self.download_manager.file_size = content_length
                                    return content_length
                            except:
                                pass
            except Exception as e:
                print(f"파일 크기 추정 오류: {str(e)}")

            # 추정 실패 시 기본값
            default_size = 1024 * 1024 * 100  # 대략 100MB로 추정
            self.download_manager.file_size = default_size
            return default_size

        # 진행률 모니터링 스레드
        def progress_monitor():
            # 파일 크기 추정
            estimated_size = estimate_file_size()
            self.download_manager.file_size = estimated_size

            # 임시 파일 크기 저장 변수 (파일 크기가 증가하지 않을 때 다운로드 완료 감지용)
            last_size = 0
            unchanged_count = 0
            progress_values = [0] * 5  # 최근 진행률 값을 저장하는 배열 (안정화용)
            progress_index = 0

            # 초기 진행률 업데이트 - 간단하게 표시
            initial_msg = f"모델 {current_model}/{total_models}"
            self.download_manager.update_progress_signal.emit(0, initial_msg)

            while self.download_manager.progress_active:
                # 취소 확인
                if self.download_manager.canceled:
                    break

                try:
                    # 이미 다운로드된 파일 크기 확인
                    current_size = 0
                    if os.path.exists(output_path):
                        current_size = os.path.getsize(output_path)
                        self.download_manager.downloaded = current_size

                    # 파일 크기가 변하지 않으면 카운트 증가 (완료 감지용)
                    if current_size == last_size and current_size > 0:
                        unchanged_count += 1
                    else:
                        unchanged_count = 0
                        last_size = current_size

                    # 진행률 계산
                    downloaded_bytes = self.download_manager.downloaded
                    total_bytes = self.download_manager.file_size

                    if total_bytes > 0:
                        # 진행률 계산 (최대 99%까지만)
                        raw_progress = min(
                            99, int(downloaded_bytes * 100 / total_bytes)
                        )

                        # 진행률 안정화 (최근 5개 값의 평균)
                        progress_values[progress_index] = raw_progress
                        progress_index = (progress_index + 1) % len(progress_values)
                        progress = sum(progress_values) // len(progress_values)

                        # 다운로드 완료로 판단되면 100%로 설정
                        if (
                            unchanged_count >= 6 and downloaded_bytes > 0
                        ):  # 3초간 크기 변화 없음
                            progress = 100

                        # 진행 상황 업데이트 - 간단하게 모델 번호만 표시
                        msg = f"모델 {current_model}/{total_models}"

                        # 시그널을 통해 진행 상황 업데이트
                        self.download_manager.update_progress_signal.emit(progress, msg)
                except Exception:
                    pass  # 모니터링 오류 무시

                # 잠시 대기
                time.sleep(0.5)

        # 모니터링 스레드 시작
        monitor_thread = threading.Thread(target=progress_monitor)
        monitor_thread.daemon = True
        monitor_thread.start()

        try:
            # 파일이 이미 존재하면 삭제
            if os.path.exists(output_path):
                try:
                    os.remove(output_path)
                except Exception as e:
                    print(f"기존 파일 삭제 실패: {str(e)}")

            # 기본 다운로드 시작 - quiet=False, fuzzy=True 옵션 추가
            gdown.download(url, output_path, quiet=False, fuzzy=True)

            # 다운로드 후 파일 존재 확인
            if not os.path.exists(output_path) or os.path.getsize(output_path) == 0:
                raise Exception("파일이 다운로드되지 않았거나 크기가 0입니다.")

            # 모니터링 중지
            self.download_manager.progress_active = False

            # 모니터링 스레드 종료 대기
            monitor_thread.join(1.0)  # 1초까지만 기다림
            return True
        except Exception as e:
            # 모니터링 중지
            self.download_manager.progress_active = False

            # 오류 발생 시 시그널 발생
            self.download_manager.download_error_signal.emit(
                f"다운로드 중 오류가 발생했습니다: {str(e)}"
            )
            return False

    # ... (나머지 코드는 동일)

    @QtCore.pyqtSlot(str, int)
    def create_progress_dialog(self, title, total_models=1):
        """다운로드 진행 상황을 표시할 대화 상자를 생성합니다."""
        # 진행 상황 대화 상자 생성
        self.progress_dialog = QProgressDialog(self)
        self.progress_dialog.setWindowTitle(title)
        self.progress_dialog.setLabelText(f"모델 1/{total_models}")

        # 진행 바 숨기기
        self.progress_dialog.setRange(0, 0)  # 무한 진행 모드로 설정(현행 마크)
        self.progress_dialog.setCancelButtonText("취소")
        self.progress_dialog.setMinimumDuration(0)  # 즉시 표시
        self.progress_dialog.setMinimumWidth(200)  # 너비 설정
        self.progress_dialog.setAutoClose(False)  # 자동 종료 방지
        self.progress_dialog.setAutoReset(False)  # 자동 리셋 방지

        # 모달이 아닌 모드로 설정 (UI가 계속 반응하도록)
        self.progress_dialog.setModal(False)

        # 취소 버튼 연결
        self.progress_dialog.canceled.connect(self.cancel_download)

        # 총 모델 수 저장
        self.total_models = total_models
        self.current_model = 1

        # 창을 표시
        self.progress_dialog.show()
        QApplication.processEvents()

    @QtCore.pyqtSlot(str)
    def update_progress_dialog_title(self, title):
        """진행 상황 대화 상자의 제목을 업데이트합니다."""
        if hasattr(self, "progress_dialog") and self.progress_dialog:
            self.progress_dialog.setWindowTitle(title)
            QApplication.processEvents()

    @QtCore.pyqtSlot(int, str)
    def update_progress_dialog(self, progress, message):
        """진행 상황 대화 상자를 업데이트합니다."""
        if hasattr(self, "progress_dialog") and self.progress_dialog:
            # 진행률을 무시하고 메시지만 업데이트
            # 단순히 "모델 n/total" 형태로 표시
            # message를 파싱하여 모델 번호만 추출
            try:
                model_info = message.split(":")[0].strip()
                self.progress_dialog.setLabelText(model_info)
            except Exception:
                self.progress_dialog.setLabelText(message)

            # UI 갱신을 위한 이벤트 처리 강제 실행
            QApplication.processEvents()

    @QtCore.pyqtSlot()
    def cancel_download(self):
        """다운로드 취소 처리를 수행합니다."""
        # 취소 신호 설정 (gdown은 직접적인 취소 방법이 없으므로 이후 작업을 중단하는 용도)
        self.download_manager.canceled = True

    @QtCore.pyqtSlot(str)
    def download_error(self, error_message):
        """다운로드 오류 처리를 수행합니다."""
        if hasattr(self, "progress_dialog") and self.progress_dialog:
            self.progress_dialog.close()
            QMessageBox.warning(self, "다운로드 오류", error_message)

    @QtCore.pyqtSlot(bool, str)
    def download_completed(self, success, message):
        """다운로드 완료 후 UI를 업데이트합니다."""
        # 진행 상황 대화 상자 닫기
        if hasattr(self, "progress_dialog") and self.progress_dialog:
            self.progress_dialog.close()
            self.progress_dialog = None

        # UI 활성화
        self.setEnabled(True)
        self.pushButton_download.setText("모델 다운로드")

        # 결과 메시지 표시
        if success:
            QMessageBox.information(self, "다운로드 완료", message)
        else:
            QMessageBox.warning(self, "다운로드 상태", message)

        # 모델 설치 상태 업데이트
        self.check_model_installation()

    def get_full_file_path(self, combo_box):
        """콤보박스에서 선택된 파일의 전체 경로를 반환합니다.

        Args:
            combo_box: 콤보박스 객체

        Returns:
            str: 파일의 전체 경로 또는 None
        """
        if combo_box.currentIndex() >= 0:
            # 데이터로 저장된 전체 경로가 있는지 확인
            file_path = combo_box.itemData(combo_box.currentIndex())
            if file_path:
                return file_path

            # 데이터가 없으면 파일명만 있는 것으로 간주
            file_name = combo_box.currentText()
            if file_name:
                # 마지막 사용 디렉토리에서 파일 찾기 시도
                possible_path = os.path.join(self.last_used_directory, file_name)
                if os.path.exists(possible_path):
                    return possible_path
        return None

    def check_model_installation(self):
        """모델 설치 상태를 확인하고 UI를 업데이트합니다."""
        # 모델 디렉토리가 없으면 생성
        if not os.path.exists(self.models_dir):
            os.makedirs(self.models_dir)
        if not os.path.exists(self.ensemble_dir):
            os.makedirs(self.ensemble_dir)

        # Single Target 모델 (RainVer1TfliteModel) 확인
        single_model_path = os.path.join(self.models_dir, "model-best.tflite")
        ver1_installed = os.path.exists(single_model_path)

        # Multi Target 모델 (RainVer2Model) 확인
        ver2_installed = os.path.exists(self.ensemble_dir)

        if ver2_installed:
            # 모든 앙상블 모델 파일이 있는지 확인
            multi_model_files = [
                f"model-best_fcst_{i}min.tflite" for i in range(10, 190, 10)
            ]
            ver2_installed = all(
                os.path.exists(os.path.join(self.ensemble_dir, f))
                for f in multi_model_files
            )

        # 선택된 모델에 따라 Predict 버튼 활성화/비활성화
        if self.radioButton_2.isChecked():  # Single Target
            self.pushButton.setEnabled(ver1_installed)
        else:  # Multi Target
            self.pushButton.setEnabled(ver2_installed)

        # 모델 상태 메시지 업데이트
        if ver1_installed and ver2_installed:
            status_msg = "모델 상태: 모든 모델이 설치되어 있습니다."
            self.pushButton_download.setEnabled(False)
        elif ver1_installed:
            status_msg = "모델 상태: Single Target 모델만 설치되어 있습니다."
            self.pushButton_download.setEnabled(True)
        elif ver2_installed:
            status_msg = "모델 상태: Multi Target 모델만 설치되어 있습니다."
            self.pushButton_download.setEnabled(True)
        else:
            status_msg = "모델 상태: 모델이 설치되어 있지 않습니다."
            self.pushButton_download.setEnabled(True)

        self.label_model_status.setText(status_msg)

        return ver1_installed, ver2_installed

    def create_prediction_dialog(self):
        """예측 진행 상황을 표시할 대화 상자를 생성합니다."""
        # 예측 상황 대화 상자 생성
        self.prediction_dialog = QDialog(self)
        self.prediction_dialog.setWindowTitle("강우 예측 진행 상황")
        self.prediction_dialog.setMinimumWidth(400)
        self.prediction_dialog.setMinimumHeight(150)

        # 레이아웃 설정
        layout = QVBoxLayout(self.prediction_dialog)

        # 상태 메시지를 표시할 레이블
        self.prediction_status_label = QLabel("예측 준비 중...")
        self.prediction_status_label.setAlignment(QtCore.Qt.AlignCenter)
        layout.addWidget(self.prediction_status_label)

        # 진행 중임을 나타내는 활동 표시기(Activity Indicator)
        self.busy_indicator = QProgressBar(self.prediction_dialog)
        self.busy_indicator.setRange(0, 0)  # 무한 진행 모드
        layout.addWidget(self.busy_indicator)

        # 닫기 버튼 (예측 중에는 비활성화됨)
        self.close_button = QPushButton("닫기", self.prediction_dialog)
        self.close_button.setEnabled(False)  # 초기에 비활성화
        self.close_button.clicked.connect(self.prediction_dialog.accept)
        layout.addWidget(self.close_button)

        # 대화 상자 표시
        self.prediction_dialog.show()
        QtWidgets.QApplication.processEvents()

    def update_prediction_status(self, message):
        """예측 상태 메시지를 업데이트합니다."""
        if hasattr(self, "prediction_dialog") and self.prediction_dialog:
            self.prediction_status_label.setText(message)
            QtWidgets.QApplication.processEvents()

    def run_prediction(self):
        """예측 버튼 클릭 시 실행되는 함수입니다."""
        # 입력 파일 경로 가져오기
        input_files = [
            self.get_full_file_path(self.comboBox),
            self.get_full_file_path(self.comboBox_2),
            self.get_full_file_path(self.comboBox_3),
            self.get_full_file_path(self.comboBox_4),
        ]

        # 출력 폴더 경로 가져오기
        output_path = self.lineEdit_5.text()

        # 파일 경로 및 출력 폴더 유효성 검사
        if not all(input_files) or not output_path:
            QMessageBox.warning(
                self, "입력 오류", "모든 입력 파일과 출력 폴더를 지정해야 합니다."
            )
            return

        if not os.path.exists(output_path):
            QMessageBox.warning(
                self, "경로 오류", f"출력 폴더 {output_path}가 존재하지 않습니다."
            )
            return

        # 예측 상태 창 생성 및 표시
        self.create_prediction_dialog()

        try:
            # 선택된 모델에 따라 적절한 함수 호출
            # 현재 날짜와 시간을 yyyymmddhhmm 형식으로 가져오기
            from datetime import datetime

            from kict_rain_forecast.services.predict import ver1_tflite_main, ver2_main

            current_time = datetime.now().strftime("%Y%m%d%H%M")

            # 날짜/시간을 포함한 출력 폴더 경로 생성
            output_path_with_time = os.path.join(output_path, current_time)

            # 폴더가 없으면 생성
            if not os.path.exists(output_path_with_time):
                os.makedirs(output_path_with_time)

            # 예측 시작 메시지 표시 (별도 창에)
            self.update_prediction_status("모델 로딩 중...")

            if self.radioButton_2.isChecked():  # Single Target
                # Ver1 TFLite 모델 사용
                model_type = "Single Target"
                self.update_prediction_status(
                    f"{model_type} 모델을 사용하여 예측 중..."
                )
                model_path = os.path.join(self.models_dir, "model-best.tflite")
                ver1_tflite_main(input_files, model_path, output_path_with_time)
            else:  # Multi Target
                # Ver2 모델 사용
                model_type = "Multi Target"
                self.update_prediction_status(
                    f"{model_type} 모델을 사용하여 예측 중..."
                )
                model_path_dir = self.ensemble_dir
                ver2_main(input_files, model_path_dir, output_path_with_time)

            # 예측 완료 메시지 표시 (별도 창에)
            self.update_prediction_status(
                f"예측 완료! 결과는 {output_path_with_time} 폴더에 저장되었습니다."
            )

            # 닫기 버튼 활성화
            if hasattr(self, "close_button"):
                self.close_button.setEnabled(True)

            # 체크박스가 체크되어 있으면 결과를 QGIS 레이어로 추가
            if self.checkBox.isChecked():
                self.update_prediction_status("결과를 QGIS 레이어로 추가 중...")
                self.add_results_to_map(output_path_with_time)
                self.update_prediction_status(
                    "완료: 결과가 QGIS 레이어에 추가되었습니다."
                )

        except Exception as e:
            # 오류 메시지 표시 (별도 창에)
            self.update_prediction_status(f"오류 발생: {str(e)}")
            if hasattr(self, "close_button"):
                self.close_button.setEnabled(True)

            # 추가로 오류 메시지 대화 상자 표시
            QMessageBox.critical(
                self, "오류 발생", f"예측 중 오류가 발생했습니다:\n{str(e)}"
            )

    def add_results_to_map(self, output_path):
        """예측 결과 파일을 QGIS 레이어로 추가합니다.

        Args:
            output_path: 결과 파일이 저장된 폴더 경로
        """
        try:
            # 결과 파일 찾기
            result_files = []

            # Single Target 모델의 결과 파일 패턴
            if self.radioButton_2.isChecked():
                # Single Target 모델은 일반적으로 "output.asc" 형태의 파일을 생성
                for i in range(10, 190, 10):
                    result_pattern = os.path.join(
                        output_path, f"QPF_REC_tflite_{i}.asc"
                    )
                    files = glob.glob(result_pattern)
                    result_files.extend(files)

            # Multi Target 모델의 결과 파일 패턴
            else:
                # Multi Target 모델은 시간별 예측 결과 파일을 생성 (10분 간격)
                for i in range(10, 190, 10):
                    result_pattern = os.path.join(output_path, f"QPF_{i}.asc")
                    files = glob.glob(result_pattern)
                    result_files.extend(files)

            if not result_files:
                QMessageBox.warning(
                    self,
                    "결과 파일 없음",
                    f"{output_path} 폴더에서 결과 파일을 찾을 수 없습니다.",
                )
                return

            # QGIS 프로젝트 가져오기
            project = QgsProject.instance()

            # 결과 파일을 레이어로 추가
            added_layers = []
            for file_path in result_files:
                file_name = os.path.basename(file_path)
                layer_name = os.path.splitext(file_name)[0]  # 확장자 제거

                # 레스터 레이어 생성
                layer = QgsRasterLayer(file_path, layer_name)

                if layer.isValid():
                    # 레이어 추가
                    project.addMapLayer(layer)
                    added_layers.append(layer_name)
                else:
                    print(f"레이어 생성 실패: {file_path}")

            # 추가된 레이어 수 표시
            if added_layers:
                QMessageBox.information(
                    self,
                    "레이어 추가 완료",
                    f"{len(added_layers)}개의 결과 레이어가 맵에 추가되었습니다.",
                )

        except Exception as e:
            QMessageBox.warning(
                self,
                "레이어 추가 오류",
                f"결과를 맵에 추가하는 중 오류가 발생했습니다:\n{str(e)}",
            )
