import copy
import os

import pytest
from pytestqt.qtbot import QtBot
from qgis.PyQt.QtCore import Qt
from qgis.PyQt.QtWidgets import QFileDialog, QMessageBox

from grm.dialogs.Weather_data import ClimateDataDialog
from grm.lib.Util import MsTitle

# Set environment variables (PyQt 실행용)
if os.environ.get("QT_QPA_PLATFORM_PLUGIN_PATH") is None:
    os.environ[
        "QT_QPA_PLATFORM_PLUGIN_PATH"
    ] = r"C:\Program Files\QGIS 3.10\apps\Qt5\plugins"
if os.environ.get("PROJ_LIB") is None:
    os.environ["PROJ_LIB"] = "C:\\PROGRA~1\\QGIS3~1.10\\share\\proj"


class TestClimateDataDialog:
    """
    ClimateDataDialog 테스트
    """

    xmltodict = {
        "GRMProject": {
            "ProjectSettings": {
                "PrecipitationDataFile": "",
                "PrecipitationInterval_min": "",
                "TemperatureMaxDataFile": "",
                "TemperatureMaxInterval_min": "",
                "TemperatureMinDataFile": "",
                "TemperatureMinInterval_min": "",
                "DaytimeLengthDataFile": "",
                "DaytimeLengthInterval_min": "",
                "DaytimeHoursRatioDataFile": "",
                "SolarRadiationDataFile": "",
                "SolarRadiationInterval_min": "",
                "SnowPackTemperatureDataFile": "",
                "SnowPackTemperatureInterval_min": "",
            }
        }
    }

    @classmethod
    def setup_class(self):
        MsTitle("ClimateDataDialog")

    @pytest.fixture(scope="class")
    def monkeyclass(self) -> pytest.MonkeyPatch:
        """
        Mocking QMessageBox

        """
        from _pytest.monkeypatch import MonkeyPatch

        mpatch = MonkeyPatch()
        mpatch.setattr(QMessageBox, "information", lambda *args: "information")
        mpatch.setattr(QMessageBox, "warning", lambda *args: "warning")
        mpatch.setattr(
            QFileDialog,
            "getOpenFileName",
            lambda *args: ("/path/mock/file/test.txt", "txt"),
        )
        yield mpatch
        mpatch.undo()

    def test_default(self, qtbot: QtBot):
        assert ClimateDataDialog()

    def test_xml_input(self, qtbot: QtBot):
        with pytest.raises(KeyError):
            ClimateDataDialog({})
        with pytest.raises(AttributeError):
            ClimateDataDialog({"GRMProject": {"ProjectSettings": ""}})

    def test_insert_precipitation(self, qtbot: QtBot, monkeyclass: pytest.MonkeyPatch):
        """
        Precipitation 정확한 데이터를 입력했을 시
        """
        xmltodict = copy.deepcopy(self.xmltodict)
        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)

        qtbot.mouseClick(dialog.ui.load_btn_PrecipitationDataFile, Qt.LeftButton)
        dialog.ui.spb_PrecipitationInterval_min.setValue(5)
        dialog.ui.spb_PrecipitationInterval_min.setValue(1)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["PrecipitationDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["PrecipitationInterval_min"] == 1
        )

    def test_not_insert_precipitation_datafile(
        self, qtbot: QtBot, monkeyclass: pytest.MonkeyPatch
    ):
        """
        Precipitation 데이터 파일을 입력하지 않았을 시

        - 경고창이 떳는지 확인
        - xmltodict가 변하지 않았는지 확인
        """
        xmltodict = copy.deepcopy(self.xmltodict)

        is_warning = False

        def set_warning(*args):
            nonlocal is_warning
            is_warning = True

        monkeyclass.setattr(QMessageBox, "warning", set_warning)

        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)
        dialog.ui.spb_PrecipitationInterval_min.setValue(5)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert is_warning
        assert xmltodict == self.xmltodict

    def test_not_insert_precipitation_interval_min(
        self, qtbot: QtBot, monkeyclass: pytest.MonkeyPatch
    ):
        """
        Precipitation interval을 입력하지 않았을 시

        - 경고창이 떳는지 확인
        - xmltodict가 변하지 않았는지 확인
        """
        xmltodict = copy.deepcopy(self.xmltodict)

        is_warning = False

        def set_warning(*args):
            nonlocal is_warning
            is_warning = True

        monkeyclass.setattr(QMessageBox, "warning", set_warning)

        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)

        qtbot.mouseClick(dialog.ui.load_btn_PrecipitationDataFile, Qt.LeftButton)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert is_warning
        assert xmltodict == self.xmltodict

    def test_insert_temperature_max(self, qtbot: QtBot):
        """
        TemperatureMax 정확한 데이터를 입력했을 시
        """
        xmltodict = copy.deepcopy(self.xmltodict)
        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)

        # 필수 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_PrecipitationDataFile, Qt.LeftButton)
        dialog.ui.spb_PrecipitationInterval_min.setValue(1)

        qtbot.mouseClick(dialog.ui.load_btn_TemperatureMaxDataFile, Qt.LeftButton)
        dialog.ui.spb_TemperatureMaxInterval_min.setValue(55)
        dialog.ui.spb_TemperatureMaxInterval_min.setValue(13)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["TemperatureMaxDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["TemperatureMaxInterval_min"]
            == 13
        )

    def test_insert_temperature_min(self, qtbot: QtBot):
        """
        TemperatureMin 정확한 데이터를 입력했을 시
        """
        xmltodict = copy.deepcopy(self.xmltodict)
        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)

        # 필수 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_PrecipitationDataFile, Qt.LeftButton)
        dialog.ui.spb_PrecipitationInterval_min.setValue(1)

        # TemperatureMin 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_TemperatureMinDataFile, Qt.LeftButton)
        dialog.ui.spb_TemperatureMinInterval_min.setValue(55)
        dialog.ui.spb_TemperatureMinInterval_min.setValue(65)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["TemperatureMinDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["TemperatureMinInterval_min"]
            == 65
        )

    def test_insert_daytime(self, qtbot: QtBot):
        """
        Daytime 정확한 데이터를 입력했을 시
        """

        xmltodict = copy.deepcopy(self.xmltodict)
        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)

        # 필수 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_PrecipitationDataFile, Qt.LeftButton)
        dialog.ui.spb_PrecipitationInterval_min.setValue(1)

        # Daytime 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_DaytimeHoursRatioDataFile, Qt.LeftButton)
        qtbot.mouseClick(dialog.ui.load_btn_DaytimeLengthDataFile, Qt.LeftButton)
        dialog.ui.spb_DaytimeLengthInterval_min.setValue(55)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["DaytimeHoursRatioDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["DaytimeLengthDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["DaytimeLengthInterval_min"]
            == 55
        )

    def test_insert_solar_radiation(self, qtbot: QtBot):
        """
        SolarRadiation 정확한 데이터를 입력했을 시
        """

        xmltodict = copy.deepcopy(self.xmltodict)
        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)

        # 필수 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_PrecipitationDataFile, Qt.LeftButton)
        dialog.ui.spb_PrecipitationInterval_min.setValue(1)

        # SolarRadiation 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_SolarRadiationDataFile, Qt.LeftButton)
        dialog.ui.spb_SolarRadiationInterval_min.setValue(55)
        dialog.ui.spb_SolarRadiationInterval_min.setValue(13)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["SolarRadiationDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["SolarRadiationInterval_min"]
            == 13
        )

    def test_insert_snow_pack_temperature(self, qtbot: QtBot):
        """
        SnowPackTemperature 정확한 데이터를 입력했을 시
        """

        xmltodict = copy.deepcopy(self.xmltodict)
        dialog = ClimateDataDialog(xmltodict)
        qtbot.addWidget(dialog)

        # 필수 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_PrecipitationDataFile, Qt.LeftButton)
        dialog.ui.spb_PrecipitationInterval_min.setValue(1)

        # SnowPackTemperature 값 입력
        qtbot.mouseClick(dialog.ui.load_btn_SnowPackTemperatureDataFile, Qt.LeftButton)
        dialog.ui.spb_SnowPackTemperatureInterval_min.setValue(55)
        dialog.ui.spb_SnowPackTemperatureInterval_min.setValue(13)
        qtbot.mouseClick(dialog.ui.btnOK, Qt.LeftButton)

        assert (
            xmltodict["GRMProject"]["ProjectSettings"]["SnowPackTemperatureDataFile"]
            == "/path/mock/file/test.txt"
        )
        assert (
            xmltodict["GRMProject"]["ProjectSettings"][
                "SnowPackTemperatureInterval_min"
            ]
            == 13
        )
